<?php

/**
 * BotAJS Git Integration - Setup Test Script
 * Run this script to verify your installation
 */

echo "=== BotAJS Git Integration - Setup Test ===\n\n";

// Test 1: PHP Version
echo "1. Testing PHP Version...\n";
$phpVersion = phpversion();
$minVersion = '7.4.0';
if (version_compare($phpVersion, $minVersion, '>=')) {
    echo "   ✓ PHP Version: {$phpVersion} (Minimum: {$minVersion})\n";
} else {
    echo "   ✗ PHP Version: {$phpVersion} (Required: {$minVersion} or higher)\n";
    exit(1);
}

// Test 2: Composer Autoloader
echo "\n2. Testing Composer Autoloader...\n";
if (file_exists(__DIR__ . '/vendor/autoload.php')) {
    require __DIR__ . '/vendor/autoload.php';
    echo "   ✓ Composer autoloader found\n";
} else {
    echo "   ✗ Composer autoloader not found. Run: composer install\n";
    exit(1);
}

// Test 3: Required Extensions
echo "\n3. Testing PHP Extensions...\n";
$requiredExtensions = ['json', 'openssl', 'curl', 'session'];
foreach ($requiredExtensions as $ext) {
    if (extension_loaded($ext)) {
        echo "   ✓ Extension '{$ext}' loaded\n";
    } else {
        echo "   ✗ Extension '{$ext}' not loaded\n";
    }
}

// Test 4: Directory Structure
echo "\n4. Testing Directory Structure...\n";
$requiredDirs = [
    'src' => __DIR__ . '/src',
    'api' => __DIR__ . '/api',
    'public' => __DIR__ . '/public',
    'storage' => __DIR__ . '/storage',
    'storage/git' => __DIR__ . '/storage/git',
    'storage/sessions' => __DIR__ . '/storage/sessions',
];

foreach ($requiredDirs as $name => $path) {
    if (is_dir($path)) {
        echo "   ✓ Directory '{$name}' exists\n";
    } else {
        echo "   ✗ Directory '{$name}' missing\n";
    }
}

// Test 5: Required Files
echo "\n5. Testing Required Files...\n";
$requiredFiles = [
    '.env' => __DIR__ . '/.env',
    'src/BotAJSClient.php' => __DIR__ . '/src/BotAJSClient.php',
    'src/GitHandler.php' => __DIR__ . '/src/GitHandler.php',
    'src/SessionManager.php' => __DIR__ . '/src/SessionManager.php',
    'api/generate-key.php' => __DIR__ . '/api/generate-key.php',
    'api/export-bot.php' => __DIR__ . '/api/export-bot.php',
    'api/import-bot.php' => __DIR__ . '/api/import-bot.php',
    'public/index.php' => __DIR__ . '/public/index.php',
];

foreach ($requiredFiles as $name => $path) {
    if (file_exists($path)) {
        echo "   ✓ File '{$name}' exists\n";
    } else {
        echo "   ✗ File '{$name}' missing\n";
    }
}

// Test 6: Environment Configuration
echo "\n6. Testing Environment Configuration...\n";
if (file_exists(__DIR__ . '/.env')) {
    $envContent = file_get_contents(__DIR__ . '/.env');
    if (strpos($envContent, 'API_KEY=') !== false) {
        echo "   ✓ API_KEY configured in .env\n";
        if (strpos($envContent, 'your-secret-api-key-here') !== false) {
            echo "   ⚠ Warning: You're using the default API key. Please change it!\n";
        }
    } else {
        echo "   ✗ API_KEY not found in .env\n";
    }
} else {
    echo "   ✗ .env file not found\n";
}

// Test 7: Git Installation
echo "\n7. Testing Git Installation...\n";
exec('git --version 2>&1', $output, $returnCode);
if ($returnCode === 0) {
    echo "   ✓ Git installed: " . trim($output[0]) . "\n";
} else {
    echo "   ✗ Git not found. Please install Git.\n";
}

// Test 8: Test Class Loading
echo "\n8. Testing Class Loading...\n";
try {
    $classes = [
        'App\\BotAJSClient',
        'App\\GitHandler',
        'App\\SessionManager',
        'App\\Request',
        'App\\Response',
        'App\\Auth',
        'App\\Config',
    ];

    foreach ($classes as $class) {
        if (class_exists($class)) {
            echo "   ✓ Class '{$class}' can be loaded\n";
        } else {
            echo "   ✗ Class '{$class}' cannot be loaded\n";
        }
    }
} catch (Exception $e) {
    echo "   ✗ Error loading classes: " . $e->getMessage() . "\n";
}

// Test 9: SSH Key Generation Test
echo "\n9. Testing SSH Key Generation...\n";
try {
    $gitHandler = new App\GitHandler('/tmp/botajs_git_test');
    $keys = $gitHandler->generateSSHKey();

    if (isset($keys['public']) && isset($keys['private'])) {
        echo "   ✓ SSH key generation successful\n";
        echo "   ✓ Public key starts with: " . substr($keys['public'], 0, 20) . "...\n";
    } else {
        echo "   ✗ SSH key generation failed\n";
    }
} catch (Exception $e) {
    echo "   ✗ Error: " . $e->getMessage() . "\n";
}

// Test 10: Storage Permissions
echo "\n10. Testing Storage Permissions...\n";
$storagePaths = [
    __DIR__ . '/storage/git',
    __DIR__ . '/storage/sessions',
];

foreach ($storagePaths as $path) {
    if (is_writable($path)) {
        echo "   ✓ '{$path}' is writable\n";
    } else {
        echo "   ⚠ '{$path}' is not writable. Run: chmod -R 755 storage/\n";
    }
}

// Summary
echo "\n=== Test Complete ===\n";
echo "\nNext Steps:\n";
echo "1. Update your .env file with a secure API_KEY\n";
echo "2. Start the development server: cd public && php -S localhost:8000\n";
echo "3. Test the API: curl -X POST http://localhost:8000/api/git/generate-key \\\n";
echo "   -H \"X-API-KEY: your-api-key\" -H \"Content-Type: application/json\" \\\n";
echo "   -d '{\"user_id\": \"test\"}'\n";
echo "\nFor more information, see README.md and QUICK_START.md\n";
